from django.utils import timezone
from .models import SystemLog, User, Buyer, Intermediary, CarType, CarItem



def add_log(action, description='', user=None, request=None):
    ip_address = None
    user_agent = None
    
    if request:
        ip_address = request.META.get('REMOTE_ADDR')
        user_agent = request.META.get('HTTP_USER_AGENT')
    
    SystemLog.objects.create(
        user=user,
        action=action,
        description=description,
        ip_address=ip_address,
        user_agent=user_agent
    )

def format_price(price):
    return f"{price:,.0f} ریال"

def create_initial_data():
    # Create admin user if not exists
    if not User.objects.filter(national_code='0000000000').exists():
        User.objects.create_superuser(
            username='admin',
            first_name='مدیر',
            last_name='سیستم',
            phone='09120000000',
            national_code='0000000000',
            shenasnameh_code='00000',
            province='تهران',
            user_type='admin'
        )
    
    # Create sample buyers
    buyers_data = [
        ('خریدار قزوین ۱', 'قزوین', '09123456789'),
        ('خریدار قزوین ۲', 'قزوین', '09123456780'),
        ('خریدار تهران ۱', 'تهران', '09123456782'),
        ('خریدار تهران ۲', 'تهران', '09123456783'),
        ('خریدار کرمان ۱', 'کرمان', '09123456785'),
        ('خریدار کرمان ۲', 'کرمان', '09123456786'),
    ]
    
    for name, province, phone in buyers_data:
        Buyer.objects.get_or_create(name=name, province=province, phone=phone)
    
    # Create sample intermediaries
    intermediaries_data = [
        ('واسطه قزوین ۱', 'قزوین', '09123456788'),
        ('واسطه قزوین ۲', 'قزوین', '09123456789'),
        ('واسطه تهران ۱', 'تهران', '09123456791'),
        ('واسطه تهران ۲', 'تهران', '09123456792'),
        ('واسطه کرمان ۱', 'کرمان', '09123456794'),
        ('واسطه کرمان ۲', 'کرمان', '09123456795'),
    ]
    
    for name, province, phone in intermediaries_data:
        Intermediary.objects.get_or_create(name=name, province=province, phone=phone)
    
    # Create car types
    car_types = ['پراید', 'پژو 405', 'پژو 206', 'پژو پارس', 'سمند', 'تیبا', 'دنا', 'کوییک']
    for type_name in car_types:
        CarType.objects.get_or_create(type_name=type_name)
    
    # Create car items
    items_data = [
        ('موتور', 'essential', 50000000, 'general'),
        ('جعبه فرمان', 'essential', 3000000, 'general'),
        ('سیستم ترمز', 'essential', 4000000, 'general'),
        ('سیستم تعلیق', 'essential', 3500000, 'general'),
        ('باتری', 'essential', 800000, 'general'),
        ('موتور پراید', 'essential', 45000000, 'پراید'),
        ('موتور پژو 206', 'essential', 60000000, 'پژو 206'),
        ('کپوت', 'optional', 1200000, 'general'),
        ('صندوق عقب', 'optional', 1000000, 'general'),
    ]
    
    for name, category, price, car_type in items_data:
        CarItem.objects.get_or_create(
            name=name,
            defaults={
                'category': category,
                'price': price,
                'car_type': car_type
            }
        )

        


def is_admin(user):
    """بررسی آیا کاربر مدیر سیستم است"""
    return user.is_authenticated and user.user_type == 'manager'

def is_manager(user):
    """بررسی آیا کاربر مدیر است"""
    return user.is_authenticated and user.user_type == 'manager'

def is_finance(user):
    """بررسی آیا کاربر مالی است"""
    return user.is_authenticated and user.user_type == 'finance'

def is_inspector(user):
    """بررسی آیا کاربر بازرس است"""
    return user.is_authenticated and user.user_type == 'inspector'