from django.db import models
from django.contrib.auth.models import AbstractUser
from django.utils import timezone

# ==========================
# کاربر سفارشی با نقش‌ها
# ==========================
class User(AbstractUser):
    USER_TYPE_CHOICES = [
        ('customer', 'کاربر عادی'),
        ('intermediary', 'واسطه'),
        ('inspector', 'بازرس'),
        ('finance', 'مالی'),
        ('manager', 'مدیر سیستم'),
    ]

    PROVINCE_CHOICES = [
        ('قزوین', 'قزوین'),
        ('تهران', 'تهران'),
        ('کرمان', 'کرمان'),
    ]

    phone = models.CharField(max_length=11, unique=True)
    national_code = models.CharField(max_length=10, unique=True)
    province = models.CharField(max_length=20, choices=PROVINCE_CHOICES)
    shenasnameh_code = models.CharField(max_length=10, blank=True, null=True)
    user_type = models.CharField(max_length=20, choices=USER_TYPE_CHOICES, default='customer')
    created_at = models.DateTimeField(default=timezone.now)

    def __str__(self):
        return f"{self.username} ({self.get_user_type_display()})"


# ==========================
# نوع خودرو
# ==========================
class CarType(models.Model):
    type_name = models.CharField(max_length=100, unique=True)
    created_at = models.DateTimeField(default=timezone.now)

    def __str__(self):
        return self.type_name


# ==========================
# خریدار / واسطه
# ==========================
class Buyer(models.Model):
    name = models.CharField(max_length=200)
    phone = models.CharField(max_length=11, blank=True, null=True)
    province = models.CharField(max_length=20)
    created_at = models.DateTimeField(default=timezone.now)

    def __str__(self):
        return self.name


class Intermediary(models.Model):
    name = models.CharField(max_length=200)
    phone = models.CharField(max_length=11, blank=True, null=True)
    province = models.CharField(max_length=20)
    created_at = models.DateTimeField(default=timezone.now)

    def __str__(self):
        return self.name


# ==========================
# خودرو
# ==========================
class Car(models.Model):
    STATUS_CHOICES = [
        ('info', 'در انتظار اطلاعات'),
        ('inspection', 'در انتظار بازرسی'),
        ('finance', 'در انتظار مالی'),
        ('manager', 'در انتظار تأیید مدیر'),
        ('approved', 'تأیید نهایی'),
        ('rejected', 'رد شده'),
    ]

    PROVINCE_CHOICES = [
        ('قزوین', 'قزوین'),
        ('تهران', 'تهران'),
        ('کرمان', 'کرمان'),
    ]

    owner_name = models.CharField(max_length=200)
    plate_number = models.CharField(max_length=20)
    vin = models.CharField(max_length=100)
    system = models.CharField(max_length=100)
    car_type = models.ForeignKey(CarType, on_delete=models.SET_NULL, null=True)
    model = models.CharField(max_length=100)
    engine_number = models.CharField(max_length=100)
    color = models.CharField(max_length=50)
    capacity = models.IntegerField()
    base_price = models.DecimalField(max_digits=15, decimal_places=2, default=0)

    user = models.ForeignKey(User, on_delete=models.CASCADE, related_name="cars")
    buyer = models.ForeignKey(Buyer, on_delete=models.SET_NULL, null=True, blank=True)
    intermediary = models.ForeignKey(Intermediary, on_delete=models.SET_NULL, null=True, blank=True)

    province = models.CharField(max_length=20, choices=PROVINCE_CHOICES)
    status = models.CharField(max_length=20, choices=STATUS_CHOICES, default='info')
    created_at = models.DateTimeField(default=timezone.now)

    def __str__(self):
        return f"{self.owner_name} - {self.plate_number}"


# ==========================
# آیتم‌های خودرو (قطعات / لوازم)
# ==========================
class CarItem(models.Model):
    CATEGORY_CHOICES = [
        ('essential', 'ضروری'),
        ('optional', 'اختیاری'),
    ]

    name = models.CharField(max_length=100, unique=True)
    category = models.CharField(max_length=20, choices=CATEGORY_CHOICES, default='essential')
    price = models.DecimalField(max_digits=15, decimal_places=2, default=0)
    car_type = models.CharField(max_length=100, default='general')
    is_active = models.BooleanField(default=True)
    created_at = models.DateTimeField(default=timezone.now)

    def __str__(self):
        return self.name


# ==========================
# پارکینگ
# ==========================
class Parking(models.Model):
    ZONE_CHOICES = [
        ('A', 'A'),
        ('B', 'B'),
        ('C', 'C'),
    ]

    PROVINCE_CHOICES = [
        ('قزوین', 'قزوین'),
        ('تهران', 'تهران'),
        ('کرمان', 'کرمان'),
    ]

    car = models.ForeignKey(Car, on_delete=models.CASCADE)
    parking_zone = models.CharField(max_length=1, choices=ZONE_CHOICES)
    parking_number = models.CharField(max_length=10)
    park_date = models.DateField()
    park_time = models.TimeField()
    province = models.CharField(max_length=20, choices=PROVINCE_CHOICES)
    created_at = models.DateTimeField(default=timezone.now)

    class Meta:
        constraints = [
            models.UniqueConstraint(
                fields=['parking_zone', 'parking_number', 'province', 'park_date'],
                name='unique_parking'
            )
        ]

    def __str__(self):
        return f"{self.car} - {self.parking_zone}{self.parking_number}"


# ==========================
# بازرسی خودرو
# ==========================
class Inspection(models.Model):
    STATUS_CHOICES = [
        ('approved', 'تایید شده'),
        ('rejected', 'رد شده'),
    ]

    car = models.ForeignKey(Car, on_delete=models.CASCADE, related_name='inspections')
    inspector = models.ForeignKey(User, on_delete=models.CASCADE, limit_choices_to={'user_type': 'inspector'})
    notes = models.TextField(blank=True, null=True)
    status = models.CharField(max_length=20, choices=STATUS_CHOICES, default='approved')
    created_at = models.DateTimeField(default=timezone.now)

    def __str__(self):
        return f"بازرسی {self.car}"


# ==========================
# آیتم‌های مفقوده در بازرسی
# ==========================
class InspectionMissingItem(models.Model):
    inspection = models.ForeignKey(Inspection, on_delete=models.CASCADE, related_name='missing_items')
    item = models.ForeignKey(CarItem, on_delete=models.CASCADE)
    item_price = models.DecimalField(max_digits=15, decimal_places=2, default=0)
    created_at = models.DateTimeField(default=timezone.now)

    def __str__(self):
        return f"{self.item.name} - {self.item_price} ریال"

# ==========================
# بررسی مالی
# ==========================
class FinanceReview(models.Model):
    car = models.ForeignKey(Car, on_delete=models.CASCADE, related_name='finance_reviews')
    finance_user = models.ForeignKey(User, on_delete=models.CASCADE, limit_choices_to={'user_type': 'finance'})
    suggested_price = models.DecimalField(max_digits=15, decimal_places=2)
    notes = models.TextField(blank=True, null=True)
    created_at = models.DateTimeField(default=timezone.now)

    def __str__(self):
        return f"مالی {self.car}"


# ==========================
# تایید مدیر
# ==========================
class ManagerApproval(models.Model):
    car = models.OneToOneField(Car, on_delete=models.CASCADE, related_name='manager_approval')
    manager = models.ForeignKey(User, on_delete=models.CASCADE, limit_choices_to={'user_type': 'manager'})
    approved = models.BooleanField(default=False)
    notes = models.TextField(blank=True, null=True)
    created_at = models.DateTimeField(default=timezone.now)

    def __str__(self):
        return f"تایید مدیر {self.car}"


# ==========================
# لاگ سیستم
# ==========================
class SystemLog(models.Model):
    user = models.ForeignKey(User, on_delete=models.SET_NULL, null=True, blank=True)
    action = models.CharField(max_length=200)
    description = models.TextField(blank=True, null=True)
    ip_address = models.GenericIPAddressField(blank=True, null=True)
    created_at = models.DateTimeField(default=timezone.now)
    user_agent = models.TextField(blank=True, null=True)

    def __str__(self):
        return f"{self.action} توسط {self.user}"


# ==========================
# آپلود فایل اکسل
# ==========================
class ExcelUpload(models.Model):
    STATUS_CHOICES = [
        ('pending', 'در انتظار پردازش'),
        ('processing', 'در حال پردازش'),
        ('completed', 'تکمیل شده'),
        ('failed', 'ناموفق'),
    ]

    user = models.ForeignKey(User, on_delete=models.CASCADE)
    file = models.FileField(upload_to='excel_uploads/')
    uploaded_at = models.DateTimeField(default=timezone.now)
    status = models.CharField(max_length=20, choices=STATUS_CHOICES, default='pending')
    total_rows = models.IntegerField(default=0)
    successful_rows = models.IntegerField(default=0)
    failed_rows = models.IntegerField(default=0)
    error_log = models.TextField(blank=True, null=True)
    
    def __str__(self):
        return f"آپلود توسط {self.user.username} - {self.uploaded_at}"

    def get_success_rate(self):
        if self.total_rows > 0:
            return (self.successful_rows / self.total_rows) * 100
        return 0